<?php

namespace Laravel\Dusk\Http\Controllers;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;

class UserController
{
    /**
     * Retrieve the authenticated user identifier and class name.
     *
     * @param  string|null  $guard
     * @return array
     */
    public function user($guard = null)
    {
        $user = Auth::guard($guard)->user();

        if (! $user) {
            return [];
        }

        return [
            'id' => $user->getAuthIdentifier(),
            'className' => get_class($user),
        ];
    }

    /**
     * Login using the given user ID / email.
     *
     * @param  string  $userId
     * @param  string|null  $guard
     * @return \Illuminate\Http\Response
     */
    public function login($userId, $guard = null)
    {
        $guard = $guard ?: config('auth.defaults.guard');

        $provider = Auth::guard($guard)->getProvider();

        $user = Str::contains($userId, '@')
                    ? $provider->retrieveByCredentials(['email' => $userId])
                    : $provider->retrieveById($userId);

        Auth::guard($guard)->login($user);

        return response(status: 204);
    }

    /**
     * Log the user out of the application.
     *
     * @param  string|null  $guard
     * @return \Illuminate\Http\Response
     */
    public function logout($guard = null)
    {
        $guard = $guard ?: config('auth.defaults.guard');

        Auth::guard($guard)->logout();

        Session::forget('password_hash_'.$guard);

        return response(status: 204);
    }

    /**
     * Get the model for the given guard.
     *
     * @param  string  $guard
     * @return string
     */
    protected function modelForGuard($guard)
    {
        $provider = config("auth.guards.{$guard}.provider");

        return config("auth.providers.{$provider}.model");
    }
}
